######################    Binomial method for the American call/put option under geometric Brownian motion   ##########
######## S: stock price
######## K: strike
######## r: interest rate
######## delta: continuous payout
######## sigma: variance
######## t: maturity
######## steps: number of time steps
american_put_bin<- function (S, K, r, delta,sigma, t, steps){
  R    = exp(r*(t/steps));
  Rinv = 1.0/R;
  
  u = exp(sigma*sqrt(t/steps));
  d = 1.0/u;
  
  p_up   = (exp((r-delta)*(t/steps))-d)/(u-d);
  p_down = 1.0-p_up;
  
  prices    = rep(0,steps+1);
  prices[1] = S*(d^steps);
  
  uu = u*u;
  
  for (i in 2:(steps+1)){
  prices[i] = uu*prices[i-1];
  }
  values = pmax(0.0, (K-prices));
  
  for (step in seq(steps,1,-1)){
  values = Rinv * ( p_up*values[2:(step+1)] + p_down*values[1:step] );
  prices = u*prices[1:step];
  values = pmax(values,K-prices);
  }
  
  return(values);
}
american_call_bin<- function(S, K, r,delta, sigma, t, steps){
  R    = exp(r*(t/steps));
  Rinv = 1.0/R;
  
  u = exp(sigma*sqrt(t/steps));
  d = 1/u;
  
  p_up   = (exp((r-delta)*(t/steps))-d)/(u-d);
  p_down = 1-p_up;
  
  prices    = rep(0,steps+1);
  prices[1] = S*(d^steps);
  
  uu = u*u;
  
  for (i in 2:(steps+1)){
    prices[i] = uu*prices[i-1];
  }
  
  call_values = pmax(0, (prices-K));
  
  for (step in seq(steps,1,-1)){
    for (i in 1:(step+1)){
      call_values[i] = (p_up*call_values[i+1]+p_down*call_values[i])*Rinv;
      prices[i] = d*prices[i+1];
      call_values[i] = pmax(call_values[i],prices[i]-K);
    }
  }
  
  call_price=call_values[1]
  return(call_price)
}
############################    Black-Scholes formula for the European option under GBM    ###########
######## st: stock price 
######## K: strike
######## r: interest rate
######## delta: continuous payout
######## tau: maturity
######## sigma: variance
integrand<-function(x,r,delta,tau,st,sigma){
  dlnorm(x,(r-delta-0.5*sigma^2)*tau + log(st),sigma*sqrt(tau))
}
BS<-function(st,K,r,delta,tau,sigma,type){
  if (type=="P"){
    value<-integrate(function(x) exp(-r*t)*(K-x)*integrand(x,r,delta,tau,st,sigma),lower=-Inf,upper=K)
    return(value$value)
  }else if (type=="C"){
    value<-integrate(function(x) exp(-r*t)*(x-K)*integrand(x,r,delta,tau,st,sigma),lower=K,upper=Inf)
    return(value$value)
  }
}
BlackScholes<-function(st,K,r,delta,tau,sigma,type){
  sigma_sqr<-sigma^2
  if (type=="C"){
    d1 = (log(st/K) + (r-delta + 0.5*sigma_sqr)*tau)/(sigma*sqrt(tau))
    d2 = d1-(sigma*sqrt(tau))
    call_price = st * exp(-delta*tau)* pnorm(d1) - K * exp(-r*tau) * pnorm(d2)
    return(call_price)
  }else if (type=="P"){
    d1 = (log(st/K) + (r-delta + 0.5*sigma_sqr)*tau)/(sigma*sqrt(tau))
    d2 = d1-(sigma*sqrt(tau))
    put_price = K * exp(-r*tau) * pnorm(-d2)-st * exp(-delta*tau)* pnorm(-d1)
    return(put_price)
  }
}


